<?php
/**
 * Plugin Name: Bulk Set Featured Image
 * Description: Sets a default featured image for specific post types if none is set.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if ( ! defined( 'ABSPATH' ) ) exit;

class Bulk_Set_Featured_Image {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_menus']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_media_assets']);
    }

    public function add_menus() {
        $post_types = get_post_types(['public' => true], 'objects');
        foreach ($post_types as $pt) {
            if ($pt->name === 'attachment') continue;
            $parent_slug = ($pt->name === 'post') ? 'edit.php' : "edit.php?post_type={$pt->name}";
            add_submenu_page($parent_slug, 'Set Permanent Image', 'Default Image', 'manage_options', "set-img-{$pt->name}", function() use ($pt) { $this->render_page($pt); });
        }

        add_options_page('Global Image Fallback', 'Global Fallback', 'manage_options', 'global-image-fallback', [$this, 'render_global_page']);
    }

    private function render_page($pt) {
        if (isset($_POST['process_bulk_image'])) {
            $this->bulk_update_posts($pt->name, $_POST['default_image_id']);
            echo '<div class="updated"><p>Database updated! All ' . $pt->label . ' now have this featured image.</p></div>';
        }

        $current_id = get_option("permanent_default_{$pt->name}");
        $this->display_form("Set Permanent Image for: {$pt->label}", $current_id, true, $pt->labels->singular_name);
    }

    public function render_global_page() {
        if (isset($_POST['save_global_image'])) {
            update_option('global_default_image_id', $_POST['default_image_id']);
            echo '<div class="updated"><p>Global fallback saved.</p></div>';
        }
        $current_id = get_option('global_default_image_id');
        $this->display_form("Global Default Fallback", $current_id, false);
    }

    private function display_form($title, $current_id, $is_bulk = false, $label = '') {
        $image_url = $current_id ? wp_get_attachment_url($current_id) : '';
        ?>
        <div class="wrap">
            <h1><?php echo $title; ?></h1>
            <form method="post">
                <div id="image-preview" style="margin-bottom: 20px; background: #eee; width: 300px; height: 200px; display: flex; align-items: center; justify-content: center;">
                    <?php if ($image_url): ?>
                        <img src="<?php echo $image_url; ?>" style="max-width: 100%; max-height: 100%;" />
                    <?php else: ?>
                        <p>No image selected</p>
                    <?php endif; ?>
                </div>
                <input type="hidden" name="default_image_id" id="default_image_id" value="<?php echo $current_id; ?>">
                <button type="button" class="button select-img">Select Image</button>
                <button type="button" class="button remove-img">Remove</button>
                
                <?php if ($is_bulk): ?>
                    <p style="color: #d63638;"><strong>Warning:</strong> Clicking "Update All" will set this image for EVERY <?php echo $label; ?> that doesn't have one. This is permanent.</p>
                    <?php submit_button('Update All Existing Posts', 'primary', 'process_bulk_image'); ?>
                <?php else: ?>
                    <p class="description">This is the emergency fallback if a post type doesn't even have a CPT-specific default.</p>
                    <?php submit_button('Save Global Fallback', 'primary', 'save_global_image'); ?>
                <?php endif; ?>
            </form>
        </div>
        <script>
            jQuery(document).ready(function($){
                var frame;
                $('.select-img').on('click', function(e){
                    e.preventDefault();
                    if (frame) { frame.open(); return; }
                    frame = wp.media({ title: 'Select Image', button: { text: 'Use this image' }, multiple: false });
                    frame.on('select', function() {
                        var attachment = frame.state().get('selection').first().toJSON();
                        $('#default_image_id').val(attachment.id);
                        $('#image-preview').html('<img src="'+attachment.url+'" style="max-width:100%;" />');
                    });
                    frame.open();
                });
                $('.remove-img').on('click', function(){
                    $('#default_image_id').val('');
                    $('#image-preview').html('<p>No image selected</p>');
                });
            });
        </script>
        <?php
    }

    private function bulk_update_posts($post_type, $image_id) {
        if (!$image_id) return;
        
        update_option("permanent_default_{$post_type}", $image_id);

        $posts = get_posts([
            'post_type' => $post_type,
            'posts_per_page' => -1,
            'fields' => 'ids',
            'meta_query' => [
                [
                    'key' => '_thumbnail_id',
                    'compare' => 'NOT EXISTS'
                ]
            ]
        ]);

        foreach ($posts as $id) {
            update_post_meta($id, '_thumbnail_id', $image_id);
        }
    }

    public function enqueue_media_assets() {
        wp_enqueue_media();
    }
}

new Bulk_Set_Featured_Image();