<?php
/**
 * Plugin Name: CPT Manager
 * Description: Create, Toggle, and Delete Custom Post Types modularly.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

class PortCPTManager {

    private $option_name = 'mcm_active_modules';
    private $modules_dir;

    public function __construct() {
        $this->modules_dir = plugin_dir_path(__FILE__) . 'modules/';
        
        add_action('plugins_loaded', [$this, 'load_modules']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_init', [$this, 'handle_create_module']);
        add_action('admin_init', [$this, 'handle_delete_module']);
        add_action('admin_init', [$this, 'handle_download_module']);
    }

    public function load_modules() {
        $active_modules = get_option($this->option_name, []);
        if (is_array($active_modules)) {
            foreach ($active_modules as $module) {
                $file = $this->modules_dir . $module;
                if (file_exists($file)) include_once $file;
            }
        }
    }

    public function add_admin_menu() {
        add_menu_page('CPT Manager', 'CPT Manager', 'manage_options', 'cpt-manager', [$this, 'settings_page_html'], 'dashicons-screenoptions', 100);
    }

    public function register_settings() {
        register_setting('mcm_settings_group', $this->option_name);
    }
	
    public function handle_download_module() {
        if (isset($_GET['mcm_download_file']) && isset($_GET['_wpnonce'])) {
            if (!wp_verify_nonce($_GET['_wpnonce'], 'mcm_download_action')) return;

            $file_to_download = sanitize_text_field($_GET['mcm_download_file']);
            $target_path = realpath($this->modules_dir . $file_to_download);

            if ($target_path && strpos($target_path, realpath($this->modules_dir)) === 0 && file_exists($target_path)) {
                header('Content-Description: File Transfer');
                header('Content-Type: application/octet-stream');
                header('Content-Disposition: attachment; filename="' . basename($target_path) . '"');
                header('Expires: 0');
                header('Cache-Control: must-revalidate');
                header('Pragma: public');
                header('Content-Length: ' . filesize($target_path));
                flush(); 
                readfile($target_path);
                exit;
            }
        }
    }


    public function handle_delete_module() {
        if (isset($_POST['mcm_delete_file']) && isset($_POST['mcm_delete_nonce'])) {
            if (!wp_verify_nonce($_POST['mcm_delete_nonce'], 'mcm_delete_action')) return;

            $file_to_delete = sanitize_text_field($_POST['mcm_delete_file']);
            $target_path = realpath($this->modules_dir . $file_to_delete);

            if ($target_path && strpos($target_path, realpath($this->modules_dir)) === 0 && file_exists($target_path)) {
                $active_modules = get_option($this->option_name, []);
                if (($key = array_search($file_to_delete, $active_modules)) !== false) {
                    unset($active_modules[$key]);
                    update_option($this->option_name, $active_modules);
                }
                unlink($target_path);
                wp_redirect(admin_url('admin.php?page=cpt-manager&deleted=1'));
                exit;
            }
        }
    }


    public function handle_create_module() {
        if (isset($_POST['mcm_create_nonce']) && wp_verify_nonce($_POST['mcm_create_nonce'], 'mcm_create_action')) {
            $singular = sanitize_text_field($_POST['cpt_singular']);
            $plural   = sanitize_text_field($_POST['cpt_plural']);
            if (empty($singular) || empty($plural)) return;

            $slug_singular = strtolower(str_replace(' ', '_', $singular));
            $slug_plural   = strtolower(str_replace(' ', '-', $plural));
            $file_name     = $slug_plural . '.php';

            $template = "<?php
/**
 * Module Name: {$plural}
 * Description: {$plural} Custom Post Type. Generated by Ryan Lyons' CPT Manager
 */

add_action('init', 'port_cpt_create_{$slug_singular}_cpt');
function port_cpt_create_{$slug_singular}_cpt() {
	\$labels = array(
		'name'               => _x( '{$plural}', 'post type general name' ),
		'singular_name'      => _x( '{$singular}', 'post type singular name' ),
		'add_new'            => _x( 'Add New', '{$slug_singular}' ),
		'add_new_item'       => __( 'Add New {$singular}' ),
		'edit_item'          => __( 'Edit {$singular}' ),
		'new_item'           => __( 'New {$singular}' ),
		'all_items'          => __( 'All {$plural}' ),
		'view_item'          => __( 'View {$singular}' ),
		'search_items'       => __( 'Search {$plural}' ),
		'not_found'          => __( 'No {$plural} found' ),
		'not_found_in_trash' => __( 'No {$plural} found in the Trash' ),
		'parent_item_colon'  => '',
		'menu_name'          => '{$plural}'
	  );
	\$rewrites = array(
	  'slug'  => '{$slug_plural}',
	  'with_front'  => true
	);
	  \$args = array(
		'labels'        => \$labels,
		'description'   => 'Displays {$plural}',
		'public'        => true,
		'menu_position' => 6,
		'menu_icon'     => 'dashicons-welcome-widgets-menus',
		'supports'      => array( 'title', 'editor', 'thumbnail', 'excerpt', 'comments' ),
		'has_archive'   => true,
		'show_in_rest'  => true,
		'rewrite'       => \$rewrites
	);
	
	register_post_type('{$slug_plural}', \$args);			
}

add_action( 'init', 'port_cpt_taxonomy_{$slug_singular}_type', 0 );
function port_cpt_taxonomy_{$slug_singular}_type() {
  \$labels = array(
    'name' => _x( '{$singular} Type', 'taxonomy general name' ),
    'singular_name' => _x( '{$singular} Type', 'taxonomy singular name' ),
    'search_items' =>  __( 'Search {$singular} Types' ),
    'all_items' => __( 'All {$singular} Types' ),
    'parent_item' => __( 'Parent {$singular} Type' ),
    'parent_item_colon' => __( 'Parent {$singular} Type:' ),
    'edit_item' => __( 'Edit {$singular} Type' ),
    'update_item' => __( 'Update {$singular} Type' ),
    'add_new_item' => __( 'Add New {$singular} Type' ),
    'new_item_name' => __( 'New {$singular} Type' ),
    'menu_name' => __( '{$singular} Types' ),
  );

  register_taxonomy('{$slug_singular}-type', '{$slug_plural}', array(
    'hierarchical' => true,
    'labels' => \$labels,
    'show_ui' => true,
    'show_admin_column' => true,
    'query_var' => true,
    'rewrite' => array( 'slug' => '{$slug_singular}-type', 'with_front' => false ),
    'show_in_rest' => true,
  ));
}
";
            file_put_contents($this->modules_dir . $file_name, $template);
            wp_redirect(admin_url('admin.php?page=cpt-manager&created=1'));
            exit;
        }
    }

    public function settings_page_html() {
        $active_modules = get_option($this->option_name, []);
        if (!is_array($active_modules)) $active_modules = [];
        $files = array_diff(scandir($this->modules_dir), array('..', '.'));
        ?>
        
        <div class="wrap">
            <h1>CPT Manager</h1>
            <?php 
                if (isset($_GET['created'])) echo '<div class="updated"><p>Module created successfully!</p></div>';
                if (isset($_GET['deleted'])) echo '<div class="error"><p>Module file deleted.</p></div>';
            ?>

            <div style="display: flex; gap: 40px; margin-top: 20px;">
                <div style="flex: 2;">
                    <h2>Active Modules</h2>
                    <form method="post" action="options.php">
                        <?php settings_fields('mcm_settings_group'); ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th style="width: 50px;">On</th>
                                    <th>Module Name</th>
                                    <th>File</th>
                                    <th style="width: 150px;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($files as $file) : if (pathinfo($file, PATHINFO_EXTENSION) == 'php') : 
                                    $data = get_file_data($this->modules_dir . $file, ['Name' => 'Module Name']);
                                    $download_url = wp_nonce_url(admin_url('admin.php?page=cpt-manager&mcm_download_file=' . $file), 'mcm_download_action');
                                ?>
                                <tr>
                                    <td><input type="checkbox" name="<?php echo $this->option_name; ?>[]" value="<?php echo $file; ?>" <?php checked(in_array($file, $active_modules)); ?>></td>
                                    <td><strong><?php echo esc_html($data['Name']); ?></strong></td>
                                    <td><code><?php echo $file; ?></code></td>
                                    <td>
                                        <a href="<?php echo $download_url; ?>" class="button button-small" style="margin-right: 5px;">Download</a>
                                        <button type="button" class="button-link-delete" style="color: #a00; vertical-align: middle;" onclick="if(confirm('Delete permanently?')) { document.getElementById('delete-<?php echo sanitize_title($file); ?>').submit(); }">Delete</button>
                                    </td>
                                </tr>
                                <?php endif; endforeach; ?>
                            </tbody>
                        </table>
                        <?php submit_button('Save Changes'); ?>
                    </form>

                    <?php foreach ($files as $file) : ?>
                        <form id="delete-<?php echo sanitize_title($file); ?>" method="post" action="" style="display:none;">
                            <?php wp_nonce_field('mcm_delete_action', 'mcm_delete_nonce'); ?>
                            <input type="hidden" name="mcm_delete_file" value="<?php echo esc_attr($file); ?>">
                        </form>
                    <?php endforeach; ?>
                </div>

                <div style="flex: 1; background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                    <h2>Create New CPT</h2>
                    <form method="post" action="">
                        <?php wp_nonce_field('mcm_create_action', 'mcm_create_nonce'); ?>
                        <p><label>Singular Name</label><br><input type="text" name="cpt_singular" class="widefat" required></p>
                        <p><label>Plural Name</label><br><input type="text" name="cpt_plural" class="widefat" required></p>
                        <p><input type="submit" class="button button-primary" value="Generate Module File"></p>
                    </form>
                </div>
            </div>
        </div>
        <?php
        if (isset($_GET['settings-updated'])) flush_rewrite_rules();
    }
}
new PortCPTManager();