<?php
/**
 * Plugin Name: Email PHP Error Notifications
 * Description: Monitors for fatal PHP errors and emails a report once a day to specified emails.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

add_action('admin_init', function() {
    register_setting('error_notifier_settings', 'error_notifier_emails', [
        'sanitize_callback' => 'sanitize_notifier_emails'
    ]);

    add_settings_section('error_notifier_main', 'Notification Settings', null, 'fatal-error-notifier');

    add_settings_field('notifier_emails_field', 'Recipient Emails', 'render_notifier_emails_field', 'fatal-error-notifier', 'error_notifier_main');
});

add_action('admin_menu', function() {
    add_options_page('Error Notifier Settings', 'Error Notifier', 'manage_options', 'fatal-error-notifier', 'render_notifier_settings_page');
});

function sanitize_notifier_emails($input) {
    $emails = explode(',', $input);
    $clean_emails = array_filter(array_map('sanitize_email', $emails));
    return implode(',', $clean_emails);
}

function render_notifier_emails_field() {
    $value = get_option('error_notifier_emails', get_option('admin_email'));
    echo '<input type="text" id="error_notifier_emails" name="error_notifier_emails" value="' . esc_attr($value) . '" class="regular-text" />';
    echo '<p class="description">Separate multiple emails with commas.</p>';
}

function render_notifier_settings_page() {
    ?>
    <div class="wrap">
        <h1>Email PHP Error Notifications</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('error_notifier_settings');
            do_settings_sections('fatal-error-notifier');
            submit_button();
            ?>
        </form>

        <hr>
        <h2>Test Configuration</h2>
        <p>Click the button below to send a test email to the recipients listed above.</p>
        <button id="send-test-error-email" class="button button-secondary">Send Test Email</button>
        <span id="test-email-response" style="margin-left: 10px; font-weight: bold;"></span>

        <script type="text/javascript">
            document.getElementById('send-test-error-email').addEventListener('click', function(e) {
                e.preventDefault();
                const responseSpan = document.getElementById('test-email-response');
                responseSpan.textContent = 'Sending...';
                responseSpan.style.color = 'inherit';

                fetch(ajaxurl, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: new URLSearchParams({
                        action: 'send_test_fatal_error_email',
                        _ajax_nonce: '<?php echo wp_create_nonce("test_error_nonce"); ?>'
                    })
                })
                .then(response => response.json())
                .then(data => {
                    responseSpan.textContent = data.message;
                    responseSpan.style.color = data.success ? 'green' : 'red';
                });
            });
        </script>
    </div>
    <?php
}

add_action('wp_ajax_send_test_fatal_error_email', function() {
    check_ajax_referer('test_error_nonce');
    
    $recipients = get_option('error_notifier_emails', get_option('admin_email'));
    $site_name = get_bloginfo('name');
    $subject = "[$site_name] Test Error Notification";
    $message = "This is a test email from your Fatal Error Notifier plugin. If you received this, your email settings are working correctly!";

    if (wp_mail($recipients, $subject, $message)) {
        wp_send_json_success(['message' => 'Test email sent successfully!']);
    } else {
        wp_send_json_error(['message' => 'Failed to send email. Check your server/SMTP settings.']);
    }
});

register_shutdown_function(function() {
    $error = error_get_last();
    $critical_types = [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR];

    if ($error !== null && in_array($error['type'], $critical_types)) {
        $existing_errors = get_option('daily_fatal_errors', []);
        $error['timestamp'] = current_time('mysql');
        $error['url'] = (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : '') . (isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '');
        $existing_errors[] = $error;
        update_option('daily_fatal_errors', $existing_errors);
    }
});

register_activation_hook(__FILE__, function() {
    if (!wp_next_scheduled('send_daily_error_report')) {
        wp_schedule_event(time(), 'daily', 'send_daily_error_report');
    }
});

register_deactivation_hook(__FILE__, function() {
    wp_clear_scheduled_hook('send_daily_error_report');
});

add_action('send_daily_error_report', function() {
    $errors = get_option('daily_fatal_errors', []);
    if (empty($errors)) return;

    $recipients = get_option('error_notifier_emails', get_option('admin_email'));
    $site_name = get_bloginfo('name');
    $subject = "[$site_name] Daily Critical Error Report";
    
    $message = "The following critical errors were detected on your site in the last 24 hours:\n\n";
    foreach ($errors as $err) {
        $message .= "--- ERROR REPORT ---\n";
        $message .= "Time: " . $err['timestamp'] . "\n";
        $message .= "Message: " . $err['message'] . "\n";
        $message .= "File: " . $err['file'] . " (Line: " . $err['line'] . ")\n";
        $message .= "URL: " . $err['url'] . "\n\n";
    }

    if (wp_mail($recipients, $subject, $message)) {
        delete_option('daily_fatal_errors');
    }
});