<?php
/**
 * Plugin Name: Outbound Link Tracker
 * Description: A basic dashboard page to see total outbound links on a site and the last 1000 in raw data.
 * Version: 1.0.0
 * Author: Ryan Lyons
 */

global $wpdb;
define('MY_LINK_TRACKER_TABLE', $wpdb->prefix . 'outbound_clicks');

register_activation_hook( __FILE__, 'my_link_tracker_activate' );
function my_link_tracker_activate() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE " . MY_LINK_TRACKER_TABLE . " (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        time datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        link_url text NOT NULL,
        source_url text NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );
}

add_action( 'wp_enqueue_scripts', 'my_link_tracker_enqueue_scripts' );
function my_link_tracker_enqueue_scripts() {
    if ( ! is_admin() ) {
        wp_enqueue_script(
            'my-link-tracker-script',
            plugins_url( 'js/tracker.js', __FILE__ ),
            array( 'jquery' ),
            '1.0',
            true
        );

        wp_localize_script( 'my-link-tracker-script', 'myLinkTracker', array(
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'my_link_tracker_nonce' )
        ) );
    }
}

add_action( 'wp_ajax_my_link_tracker_log', 'my_link_tracker_log_click' );
add_action( 'wp_ajax_nopriv_my_link_tracker_log', 'my_link_tracker_log_click' );
function my_link_tracker_log_click() {
    if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'my_link_tracker_nonce' ) ) {
        wp_send_json_error( 'Nonce verification failed.' );
        wp_die();
    }

    $link_url = isset( $_POST['link_url'] ) ? esc_url_raw( $_POST['link_url'] ) : '';
    $source_url = isset( $_POST['source_url'] ) ? esc_url_raw( $_POST['source_url'] ) : '';

    if ( empty( $link_url ) || empty( $source_url ) ) {
        wp_send_json_error( 'Missing data.' );
        wp_die();
    }

    global $wpdb;
    $table_name = MY_LINK_TRACKER_TABLE;

    $wpdb->insert(
        $table_name,
        array(
            'link_url'   => $link_url,
            'source_url' => $source_url
        ),
        array(
            '%s',
            '%s'
        )
    );

    wp_send_json_success( 'Click tracked successfully.' );
    wp_die();
}

add_action( 'admin_menu', 'my_link_tracker_admin_menu' );
function my_link_tracker_admin_menu() {
    add_menu_page(
        'Outbound Clicks',
        'Link Tracker',
        'manage_options',
        'my-link-tracker-dashboard',
        'my_link_tracker_dashboard_page',
        'dashicons-external',
        2
    );
}

function my_link_tracker_dashboard_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    global $wpdb;
    $table_name = MY_LINK_TRACKER_TABLE;

    $today = date('Y-m-d'); 
    
    $one_week_ago = date('Y-m-d', strtotime('-7 days')); 

    $date_from = isset( $_GET['date_from'] ) && ! empty( $_GET['date_from'] ) ? sanitize_text_field( $_GET['date_from'] ) : $one_week_ago;
    $date_to = isset( $_GET['date_to'] ) && ! empty( $_GET['date_to'] ) ? sanitize_text_field( $_GET['date_to'] ) : $today;

    $where_clause = '';
    $sql_params = array();

    if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
        $date_to_inclusive = $date_to . ' 23:59:59'; 
        
        $where_clause = ' WHERE time BETWEEN %s AND %s';
        $sql_params[] = $date_from;
        $sql_params[] = $date_to_inclusive;
    }
    
    $sql = "SELECT link_url, COUNT(link_url) as total_clicks 
            FROM $table_name
            $where_clause
            GROUP BY link_url
            ORDER BY total_clicks DESC";

    $prepared_sql = $wpdb->prepare( $sql, $sql_params );
    $results = $wpdb->get_results( $prepared_sql, ARRAY_A );

    echo '<div class="wrap">';
    echo '<h1>Outbound Links</h1>';
    
    echo '<form method="get" action="">';
    echo '<input type="hidden" name="page" value="my-link-tracker-dashboard" />'; // Important
    echo '<label for="date_from">From Date:</label>';
    echo '<input type="date" id="date_from" name="date_from" value="' . esc_attr( $date_from ) . '">';
    echo ' ';
    echo '<label for="date_to">To Date:</label>';
    echo '<input type="date" id="date_to" name="date_to" value="' . esc_attr( $date_to ) . '">';
    echo ' ';
    echo '<input type="submit" class="button button-primary" value="Filter Clicks">';
    //INCOMPLETE
    // echo ' ';
    // echo '<a href="' . esc_url( admin_url( 'admin-post.php?action=my_link_tracker_export' ) ) . '" class="button button-secondary">Download Lifetime Data (CSV)</a>';
    // echo ' '; 
    // echo '<a href="' . esc_url( admin_url( 'admin-post.php?action=my_link_tracker_export_totals' ) ) . '" class="button button-secondary">Download Aggregated Totals (CSV)</a>';
    
    echo ' <a href="' . esc_url( admin_url( 'admin.php?page=my-link-tracker-dashboard' ) ) . '" class="button">Reset to Last 7 Days</a>';
    
    echo '</form>';
    
    echo '<hr>';

    if ( empty( $results ) ) {
        echo '<p>No outbound clicks found for the selected date range.</p>';
        echo '</div>';
        return;
    }

    echo '<table class="wp-list-table widefat fixed striped">';
    echo '<thead>';
    echo '<tr><th>Outbound Link URL</th><th>Total Clicks</th></tr>';
    echo '</thead>';
    echo '<tbody>';

    foreach ( $results as $row ) {
        echo '<tr>';
        echo '<td><a href="' . esc_url( $row['link_url'] ) . '" target="_blank">' . esc_url( $row['link_url'] ) . '</a></td>';
        echo '<td><strong>' . absint( $row['total_clicks'] ) . '</strong></td>';
        echo '</tr>';
    }

    echo '</tbody>';
    echo '</table>';
    
    $recent_results = $wpdb->get_results(
        "SELECT time, link_url, source_url
         FROM $table_name
         ORDER BY time DESC
         LIMIT 1000",
        ARRAY_A
    );
    
    echo '<h2>Last 1000 Individual Clicks (Raw Data)</h2>';

    if ( empty( $recent_results ) ) {
        echo '<p>No recent individual clicks to display.</p>';
    } else {
        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead>';
        echo '<tr><th>Timestamp</th><th>Link Clicked</th><th>Source Page</th></tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ( $recent_results as $row ) {
            $click_time = date( 'Y-m-d H:i:s', strtotime( $row['time'] ) );
            
            echo '<tr>';
            echo '<td>' . esc_html( $click_time ) . '</td>';
            echo '<td><a href="' . esc_url( $row['link_url'] ) . '" target="_blank">' . esc_url( $row['link_url'] ) . '</a></td>';
            echo '<td><a href="' . esc_url( $row['source_url'] ) . '" target="_blank">' . esc_url( $row['source_url'] ) . '</a></td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
    }
    
    echo '</div>';
}

//Incomplete Downloads in CSV for Raw Data Lifetime and Aggregated Totals
//add_action( 'admin_post_my_link_tracker_export', 'my_link_tracker_export_data' );
//add_action( 'admin_post_my_link_tracker_export_totals', 'my_link_tracker_export_totals' );

function my_link_tracker_export_data() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'You do not have sufficient permissions to access this page.' );
    }

    global $wpdb;
    $table_name = MY_LINK_TRACKER_TABLE;
    
    $results = $wpdb->get_results( "SELECT time, link_url, source_url FROM $table_name ORDER BY time ASC", ARRAY_A );

    if ( empty( $results ) ) {
        wp_die( 'No data found to export.' );
    }

    $csv_output = "Timestamp,Outbound Link URL,Source Page URL\n";

    foreach ( $results as $row ) {
        $csv_output .= '"' . str_replace('"', '""', $row['time']) . '",';
        $csv_output .= '"' . str_replace('"', '""', $row['link_url']) . '",';
        $csv_output .= '"' . str_replace('"', '""', $row['source_url']) . '"';
        $csv_output .= "\n";
    }

    $filename = 'link-tracker-lifetime-export-' . date('Ymd-His') . '.csv';
    header('Content-Type: text/csv');
    header("Content-Disposition: attachment; filename=$filename");
    header('Pragma: no-cache');
    header('Expires: 0');
    
    echo $csv_output;
    
    wp_die(); 
}

function my_link_tracker_export_totals() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'You do not have sufficient permissions to access this page.' );
    }

    global $wpdb;
    $table_name = MY_LINK_TRACKER_TABLE;

    $results = $wpdb->get_results(
        "SELECT link_url, COUNT(link_url) as total_clicks
         FROM $table_name
         GROUP BY link_url
         ORDER BY total_clicks DESC",
        ARRAY_A
    );

    if ( empty( $results ) ) {
        wp_die( 'No aggregated data found to export.' );
    }

    $csv_output = "Total Clicks,Outbound Link URL\n";

    foreach ( $results as $row ) {
        $csv_output .= '"' . absint( $row['total_clicks'] ) . '",';
        $csv_output .= '"' . str_replace('"', '""', $row['link_url']) . '"';
        $csv_output .= "\n";
    }

    $filename = 'link-tracker-aggregated-totals-' . date('Ymd-His') . '.csv';
    header('Content-Type: text/csv');
    header("Content-Disposition: attachment; filename=$filename");
    header('Pragma: no-cache');
    header('Expires: 0');
    
    echo $csv_output;
    
    wp_die(); 
}