<?php
/**
 * Plugin Name: Site Lockdown
 * Description: Password protects the front-end and allows admins to block specific roles from the backend.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

add_action('admin_menu', 'slock_add_settings_menu');
function slock_add_settings_menu() {
    add_options_page('Site Lockdown Settings', 'Site Lockdown', 'manage_options', 'site-lockdown', 'slock_settings_page');
}

add_action('admin_enqueue_scripts', 'slock_enqueue_admin_assets');
function slock_enqueue_admin_assets($hook) {
    if ($hook !== 'settings_page_site-lockdown') return;
    wp_enqueue_media();
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
    add_action('admin_footer', function() {
        ?>
        <script>
        jQuery(document).ready(function($){
            $('.slock-color-field').wpColorPicker();
            $('.slock-upload-button').click(function(e) {
                e.preventDefault();
                var custom_uploader = wp.media({
                    title: 'Select Logo',
                    button: { text: 'Use this logo' },
                    multiple: false
                }).on('select', function() {
                    var attachment = custom_uploader.state().get('selection').first().toJSON();
                    $('.slock-logo-url').val(attachment.url);
                    $('.slock-preview-img').attr('src', attachment.url).show();
                }).open();
            });
        });
        </script>
        <?php
    });
}

add_action('admin_init', 'slock_register_settings');
function slock_register_settings() {
    register_setting('slock_settings_group', 'slock_password');
    register_setting('slock_settings_group', 'slock_message');
    register_setting('slock_settings_group', 'slock_bg_color');
    register_setting('slock_settings_group', 'slock_logo');
    register_setting('slock_settings_group', 'slock_blocked_roles');

    add_settings_section('slock_main_section', 'Design & Security', null, 'site-lockdown');

    add_settings_field('slock_password', 'Access Password', function() {
        echo '<input type="password" name="slock_password" value="'.esc_attr(get_option('slock_password')).'" />';
		echo '<p class="description">If no password is set, the site will remain unlocked</p>';
    }, 'site-lockdown', 'slock_main_section');

    add_settings_field('slock_logo', 'Logo Image', 'slock_logo_callback', 'site-lockdown', 'slock_main_section');

    add_settings_field('slock_bg_color', 'Background Color', function() {
        echo '<input type="text" name="slock_bg_color" value="'.esc_attr(get_option('slock_bg_color', '#ffffff')).'" class="slock-color-field" />';
    }, 'site-lockdown', 'slock_main_section');

    add_settings_field('slock_blocked_roles', 'Block Access for Roles', 'slock_roles_callback', 'site-lockdown', 'slock_main_section');

    add_settings_field('slock_message', 'Display Message', function() {
        echo '<textarea name="slock_message" rows="5" cols="50">'.esc_textarea(get_option('slock_message')).'</textarea>';
    }, 'site-lockdown', 'slock_main_section');
}

function slock_logo_callback() {
    $logo = get_option('slock_logo');
    echo '<input type="hidden" name="slock_logo" class="slock-logo-url" value="'.esc_attr($logo).'">';
    echo '<img src="'.esc_attr($logo).'" class="slock-preview-img" style="max-width:150px; display:'.($logo ? 'block' : 'none').'; margin-bottom:10px;">';
    echo '<button class="slock-upload-button button">Upload/Select Logo</button>';
}

function slock_roles_callback() {
    global $wp_roles;
    $blocked_roles = get_option('slock_blocked_roles', array());
    if (!is_array($blocked_roles)) $blocked_roles = array();

    foreach ($wp_roles->roles as $role_key => $role_data) {
        if ($role_key === 'administrator') continue; // Don't allow blocking yourself
        $checked = in_array($role_key, $blocked_roles) ? 'checked' : '';
        echo '<label style="display:block;"><input type="checkbox" name="slock_blocked_roles[]" value="'.$role_key.'" '.$checked.'> '.$role_data['name'].'</label>';
    }
    echo '<p class="description">Checked roles will be treated as logged-out users and blocked from the backend.</p>';
}

function slock_settings_page() {
    ?>
    <div class="wrap">
        <h1>Site Lockdown Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('slock_settings_group');
            do_settings_sections('site-lockdown');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

add_action('init', 'slock_global_lockdown_check');
function slock_global_lockdown_check() {
    if (is_admin() && !defined('DOING_AJAX')) {
        slock_restrict_admin_access();
    }
}

add_action('template_redirect', 'slock_check_lockdown_front');
function slock_check_lockdown_front() {
    if (!session_id()) session_start();
    $password = get_option('slock_password');
    if (empty($password)) return;

    $is_blocked_role = false;
    if (is_user_logged_in()) {
        $user = wp_get_current_user();
        $blocked_roles = get_option('slock_blocked_roles', array());
        foreach ($user->roles as $role) {
            if (in_array($role, $blocked_roles)) {
                $is_blocked_role = true;
                break;
            }
        }
    }

    if ((is_user_logged_in() && !$is_blocked_role) || is_login() || (isset($_SESSION['slock_access']) && $_SESSION['slock_access'] === $password)) {
        return;
    }

    if (isset($_POST['slock_submit_pass'])) {
        if ($_POST['slock_entered_pass'] === $password) {
            $_SESSION['slock_access'] = $password;
            wp_safe_redirect($_SERVER['REQUEST_URI']);
            exit;
        } else { $error = "Incorrect password."; }
    }
    slock_display_lock_screen($error ?? '');
    exit;
}

function slock_restrict_admin_access() {
    if (current_user_can('administrator')) return;

    $user = wp_get_current_user();
    $blocked_roles = get_option('slock_blocked_roles', array());
    
    foreach ($user->roles as $role) {
        if (in_array($role, $blocked_roles)) {
            wp_redirect(home_url());
            exit;
        }
    }
}

function slock_display_lock_screen($error) {
    $message = get_option('slock_message');
    $bg_color = get_option('slock_bg_color', '#ffffff');
    $logo = get_option('slock_logo');
    
    $hex = str_replace('#', '', $bg_color);
    $r = hexdec(substr($hex, 0, 2)); $g = hexdec(substr($hex, 2, 2)); $b = hexdec(substr($hex, 4, 2));
    $brightness = (($r * 299) + ($g * 587) + ($b * 114)) / 1000;
    $text_color = ($brightness > 125) ? '#000000' : '#ffffff';
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>Access Protected</title>
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <style>
            body { background-color: <?php echo esc_attr($bg_color); ?>; color: <?php echo $text_color; ?>; height: 100vh; display: flex; align-items: center; justify-content: center; font-family: -apple-system, sans-serif; margin: 0; }
            .container { text-align: center; max-width: 500px; padding: 20px; width: 100%; }
            .logo { max-width: 200px; margin-bottom: 20px; }
            .message { font-size: 1.25rem; margin-bottom: 25px; line-height: 1.5; }
            input[type="password"] { padding: 12px; width: 80%; max-width: 300px; border: 1px solid #ccc; border-radius: 4px; margin-bottom: 10px; font-size: 16px; }
            input[type="submit"] { padding: 12px; width: 80%; max-width: 300px; background: <?php echo $text_color; ?>; color: <?php echo $bg_color; ?>; border: none; border-radius: 4px; cursor: pointer; font-weight: bold; font-size: 16px; }
            .error { color: #ff4d4d; font-weight: bold; margin-bottom: 10px; }
        </style>
    </head>
    <body>
        <div class="container">
            <?php if ($logo): ?>
                <img src="<?php echo esc_url($logo); ?>" class="logo">
            <?php endif; ?>
            <div class="message"><?php echo nl2br(esc_html($message)); ?></div>
            <?php if ($error) echo "<p class='error'>$error</p>"; ?>
            <form method="post">
                <input type="password" name="slock_entered_pass" placeholder="Password" autofocus required>
                <br>
                <input type="submit" name="slock_submit_pass" value="Unlock Site">
            </form>
        </div>
    </body>
    </html>
    <?php
}