<?php
/**
 * Plugin Name: One By One Replace
 * Description: Search for strings or Regex with optional case sensitivity and replace specific occurrences within any CPT.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

add_action('admin_menu', 'obor_add_menu');
function obor_add_menu() {
    add_management_page('One-By-One Replace', 'One-By-One Replace', 'manage_options', 'one-by-one-replace', 'obor_render_page');
}

function obor_render_page() {
    $post_types = get_post_types(array('public' => true), 'objects');
    ?>
    <style>
        .obor-container { display: flex; flex-wrap: wrap; gap: 20px; margin-top: 20px; align-items: flex-start; }
        .obor-sidebar { flex: 0 0 30%; min-width: 300px; position: sticky; top: 40px; }
        .obor-results { flex: 1; min-width: 500px; }
        .obor-sidebar .card { padding: 15px; margin: 0; }
        .obor-sidebar table.form-table th { width: 100px; padding: 10px 0; }
        .obor-sidebar table.form-table td { padding: 10px 0; }
        .obor-sidebar input[type="text"], .obor-sidebar select { width: 100%; }
        
        @media (max-width: 800px) {
            .obor-sidebar { flex: 1 0 100%; position: static; }
            .obor-results { flex: 1 0 100%; }
        }
        .match-tag { padding: 2px 1px; border-radius: 2px; }
    </style>

    <div class="wrap">
        <h1>One By One Replace</h1>
        
        <div class="obor-container">
            <div class="obor-sidebar">
                <div class="card">
                    <form method="GET">
                        <input type="hidden" name="page" value="one-by-one-replace">
                        <table class="form-table">
                            <tr>
                                <td><strong>Search Pattern:</strong><br>
                                <input name="search_string" type="text" value="<?php echo esc_attr($_GET['search_string'] ?? ''); ?>" required></td>
                            </tr>
                            <tr>
                                <td><strong>In Post Type:</strong><br>
                                <select name="target_post_type">
                                    <?php foreach ($post_types as $pt) : ?>
                                        <option value="<?php echo $pt->name; ?>" <?php selected($_GET['target_post_type'] ?? '', $pt->name); ?>><?php echo $pt->label; ?></option>
                                    <?php endforeach; ?>
                                </select></td>
                            </tr>
                            <tr>
                                <td><strong>Replace With:</strong><br>
                                <input name="replace_string" type="text" value="<?php echo esc_attr($_GET['replace_string'] ?? ''); ?>" required></td>
                            </tr>
                            <tr>
                                <td>
                                    <label><input type="checkbox" name="use_regex" value="1" <?php checked($_GET['use_regex'] ?? 0, 1); ?>> Use Regex</label><br>
                                    <label><input type="checkbox" name="case_sensitive" value="1" <?php checked($_GET['case_sensitive'] ?? 0, 1); ?>> Case Sensitive</label>
                                </td>
                            </tr>
                        </table>
                        <p class="submit"><input type="submit" name="search_now" class="button button-primary" style="width:100%;" value="Find All Occurrences"></p>
                    </form>
                </div>
            </div>

            <div class="obor-results">
                <?php if (isset($_GET['search_now'])) obor_handle_search(); ?>
            </div>
        </div>
    </div>
    <?php
}

function obor_handle_search() {
    global $wpdb;
    $search = $_GET['search_string'];
    $replace = $_GET['replace_string'];
    $post_type = $_GET['target_post_type'];
    $use_regex = isset($_GET['use_regex']);
    $case_sensitive = isset($_GET['case_sensitive']);

    if ($use_regex) {
        $operator = 'REGEXP';
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT ID, post_title, post_content FROM $wpdb->posts WHERE post_type = %s AND post_status = 'publish' AND (post_content $operator %s OR post_title $operator %s)",
            $post_type, $search, $search
        ));
    } else {
        $operator = $case_sensitive ? 'LIKE BINARY' : 'LIKE';
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT ID, post_title, post_content FROM $wpdb->posts WHERE post_type = %s AND post_status = 'publish' AND (post_content $operator %s OR post_title $operator %s)",
            $post_type, '%' . $wpdb->esc_like($search) . '%', '%' . $wpdb->esc_like($search) . '%'
        ));
    }

    if (!$results) { echo "<div class='notice notice-warning'><p>No matches found.</p></div>"; return; }

    echo "<div style='margin-bottom:15px; background: #fff; padding: 10px; border: 1px solid #ccd0d4;'>
            <button type='button' id='bulk-replace-btn' class='button button-secondary'>Replace All Selected</button>
            <span id='bulk-status' style='margin-left:10px; font-weight:600;'></span>
          </div>";

    echo "<table class='wp-list-table widefat fixed striped'>
            <thead>
                <tr>
                    <th class='check-column'><input id='cb-select-all' type='checkbox'></th>
                    <th>Post Title</th>
                    <th>Context</th>
                    <th style='width:100px;'>Action</th>
                </tr>
            </thead>
            <tbody>";

    foreach ($results as $post) {
        foreach (['title', 'content'] as $field_type) {
            $text = ($field_type === 'title') ? $post->post_title : $post->post_content;
            $found_items = [];
            
            if ($use_regex) {
                $pattern = '/' . str_replace('/', '\/', $search) . '/' . ($case_sensitive ? '' : 'i') . 'u';
                preg_match_all($pattern, $text, $matches, PREG_OFFSET_CAPTURE);
                $found_items = $matches[0] ?? [];
            } else {
                $pos = 0;
                $strpos_func = $case_sensitive ? 'mb_strpos' : 'mb_stripos';
                while (($pos = $strpos_func($text, $search, $pos)) !== false) {
                    $found_items[] = [mb_substr($text, $pos, mb_strlen($search)), $pos];
                    $pos += mb_strlen($search);
                }
            }

            foreach ($found_items as $index => $item) {
                $val = $item[0];
                $pos = $item[1];
                $start = max(0, $pos - 35);
                $snippet = mb_substr($text, $start, 80);
                obor_render_row($post->ID, $post->post_title, $val, $replace, $field_type, $index, $snippet, $pos - $start, $use_regex, $case_sensitive, $search);
            }
        }
    }
    echo "</tbody></table>";
}

function obor_render_row($post_id, $title, $found_val, $replace, $type, $index, $snippet, $relative_pos, $is_regex, $cs, $raw_search) {
    $before = mb_substr($snippet, 0, $relative_pos);
    $after = mb_substr($snippet, $relative_pos + mb_strlen($found_val));
    $highlighted = esc_html($before) . "<b class='match-tag' style='color:red; background:yellow;'>" . esc_html($found_val) . "</b>" . esc_html($after);

    echo "<tr>
            <th class='check-column'><input type='checkbox' class='post-checkbox'></th>
            <td>" . esc_html($title) . "</td>
            <td><code>...{$highlighted}...</code></td>
            <td>
                <button class='button obor-replace-btn' 
                    data-postid='{$post_id}' data-type='{$type}' data-index='{$index}' 
                    data-search='".esc_attr($is_regex ? $raw_search : $found_val)."' data-replace='".esc_attr($replace)."'
                    data-isregex='".($is_regex ? '1' : '0')."' data-cs='".($cs ? '1' : '0')."'>Replace</button>
            </td>
          </tr>";
}

add_action('wp_ajax_obor_perform_replace', function() {
    check_ajax_referer('obor_nonce', 'security');
    $id = intval($_POST['post_id']);
    $post = get_post($id);
    $text = ($_POST['type'] === 'title') ? $post->post_title : $post->post_content;
    $search = $_POST['search'];
    $replace = $_POST['replace'];
    $index = intval($_POST['index']);

    if ($_POST['isregex'] === '1') {
        $pattern = '/' . str_replace('/', '\/', $search) . '/' . ($_POST['cs'] === '1' ? '' : 'i') . 'u';
        $count = 0;
        $new_text = preg_replace_callback($pattern, function($m) use (&$count, $index, $replace) {
            return ($count++ === $index) ? $replace : $m[0];
        }, $text);
    } else {
        $pos = -1;
        $func = ($_POST['cs'] === '1') ? 'mb_strpos' : 'mb_stripos';
        for ($i = 0; $i <= $index; $i++) { $pos = $func($text, $search, $pos + 1); if ($pos === false) break; }
        $new_text = ($pos !== false) ? mb_substr($text, 0, $pos) . $replace . mb_substr($text, $pos + mb_strlen($search)) : $text;
    }

    $args = ['ID' => $id];
    ($_POST['type'] === 'title') ? $args['post_title'] = $new_text : $args['post_content'] = $new_text;
    wp_update_post($args);
    wp_send_json_success();
});

add_action('admin_footer', function() {
    ?>
    <script>
    jQuery(document).ready(function($) {
        $('#cb-select-all').on('change', function() { $('.post-checkbox').prop('checked', $(this).prop('checked')); });

        $(document).on('click', '.obor-replace-btn', function() { runReplace($(this), $(this).closest('tr')); });

        function runReplace(btn, row) {
            var replaceText = btn.data('replace');
            return $.post(ajaxurl, {
                action: 'obor_perform_replace',
                post_id: btn.data('postid'),
                type: btn.data('type'),
                index: btn.data('index'),
                search: btn.data('search'),
                replace: replaceText,
                isregex: btn.data('isregex'),
                cs: btn.data('cs'),
                security: '<?php echo wp_create_nonce("obor_nonce"); ?>'
            }, function(res) {
                if(res.success) {
                    row.find('.match-tag').text(replaceText).css({'color': 'green', 'background': '#d4edda'});
                    btn.text('Done').prop('disabled', true);
                }
            });
        }

        $('#bulk-replace-btn').on('click', async function() {
            var selectedRows = $('.post-checkbox:checked').closest('tr');
            if (selectedRows.length === 0) return alert('Select items.');
            $(this).prop('disabled', true);
            for (var i = 0; i < selectedRows.length; i++) {
                var row = $(selectedRows[i]);
                $('#bulk-status').text('Updating ' + (i+1) + '/' + selectedRows.length);
                await runReplace(row.find('.obor-replace-btn'), row);
            }
            $('#bulk-status').text('Complete!');
            $(this).prop('disabled', false);
        });
    });
    </script>
    <?php
});