<?php
/**
 * Plugin Name: Post Powers
 * Description: One-click duplication and post-type switching for Posts, Pages, and CPTs.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if ( ! defined( 'ABSPATH' ) ) exit;

function post_powers_add_clone_link( $actions, $post ) {
    if ( current_user_can( 'edit_posts' ) ) {
        $actions['clone'] = '<a href="' . wp_nonce_url( admin_url( 'admin-post.php?action=post_powers_duplicate&post=' . $post->ID ), 'post_powers_duplicate_' . $post->ID ) . '" title="Clone this item">Clone</a>';
    }
    return $actions;
}
add_filter( 'post_row_actions', 'post_powers_add_clone_link', 10, 2 );
add_filter( 'page_row_actions', 'post_powers_add_clone_link', 10, 2 );

function post_powers_handle_duplicate() {
    if ( ! isset( $_GET['post'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'post_powers_duplicate_' . (int)$_GET['post'] ) ) return;
    
    $post_id = (int) $_GET['post'];
    $post = get_post( $post_id );
    $new_post_id = wp_insert_post( array(
        'post_title' => $post->post_title . ' (Copy)',
        'post_content' => $post->post_content,
        'post_status' => 'draft',
        'post_type' => $post->post_type,
        'post_author' => get_current_user_id(),
    ) );

    if ( $new_post_id ) {
        $meta = get_post_custom( $post_id );
        foreach ( $meta as $key => $values ) {
            foreach ( $values as $value ) add_post_meta( $new_post_id, $key, $value );
        }
    }
    wp_redirect( admin_url( 'edit.php?post_type=' . $post->post_type ) );
    exit;
}
add_action( 'admin_post_post_powers_duplicate', 'post_powers_handle_duplicate' );

function post_powers_add_column( $columns ) {
    $columns['pp_type'] = 'Type';
    return $columns;
}
add_filter( 'manage_posts_columns', 'post_powers_add_column' );
add_filter( 'manage_pages_columns', 'post_powers_add_column' );

function post_powers_column_display( $column_name, $post_id ) {
    if ( $column_name === 'pp_type' ) {
        echo get_post_type_object( get_post_type( $post_id ) )->labels->singular_name;
    }
}
add_action( 'manage_posts_custom_column', 'post_powers_column_display', 10, 2 );
add_action( 'manage_pages_custom_column', 'post_powers_column_display', 10, 2 );

function post_powers_render_quick_edit( $column_name, $post_type ) {
    if ( $column_name !== 'pp_type' ) return;
    $post_types = get_post_types( array( 'public' => true ), 'objects' );
    ?>
    <fieldset class="inline-edit-col-right">
        <div class="inline-edit-col">
            <label>
                <span class="title">Convert to</span>
                <select name="post_powers_new_type">
                    <option value="-1">— No Change —</option>
                    <?php foreach ( $post_types as $type ) : ?>
                        <option value="<?php echo esc_attr( $type->name ); ?>"><?php echo esc_html( $type->label ); ?></option>
                    <?php endforeach; ?>
                </select>
            </label>
        </div>
    </fieldset>
    <?php
}
add_action( 'quick_edit_custom_box', 'post_powers_render_quick_edit', 10, 2 );
add_action( 'bulk_edit_custom_box', 'post_powers_render_quick_edit', 10, 2 );

function post_powers_save_bulk_quick_edit( $post_id ) {
    if ( ! isset( $_REQUEST['post_powers_new_type'] ) || $_REQUEST['post_powers_new_type'] === '-1' ) return;
    
    if ( ! current_user_can( 'edit_post', $post_id ) ) return;

    set_post_type( $post_id, sanitize_text_field( $_REQUEST['post_powers_new_type'] ) );
}
add_action( 'save_post', 'post_powers_save_bulk_quick_edit' );