<?php
/**
 * Plugin Name: URL Redirects
 * Description: Create and Edit 301 redirects.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

register_activation_hook(__FILE__, 'msr_create_table');
function msr_create_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'custom_redirects';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        source_url varchar(255) NOT NULL,
        target_url varchar(255) NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

add_action('admin_menu', 'msr_add_admin_menu');
function msr_add_admin_menu() {
    add_menu_page('Redirects', 'Redirects', 'manage_options', 'redirects', 'msr_settings_page', 'dashicons-randomize');
}

function msr_settings_page() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'custom_redirects';
    $message = '';
    $page_slug = 'redirects';

    if (isset($_POST['bulk_action']) && $_POST['bulk_action'] === 'delete' && !empty($_POST['redirect_ids'])) {
        check_admin_referer('bulk_redirects_nonce');
        $ids = array_map('intval', $_POST['redirect_ids']);
        $ids_string = implode(',', $ids);
        $wpdb->query("DELETE FROM $table_name WHERE id IN ($ids_string)");
        $message = '<div class="updated"><p>Selected redirects deleted.</p></div>';
    }

    if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
        if (check_admin_referer('delete_redirect_' . $_GET['id'])) {
            $wpdb->delete($table_name, ['id' => intval($_GET['id'])]);
            $message = '<div class="updated"><p>Redirect deleted.</p></div>';
        }
    }

    if (isset($_POST['update_redirect'])) {
        check_admin_referer('edit_redirect_nonce');
        $id = intval($_POST['redirect_id']);
        $source = str_replace(' ', '%20', sanitize_text_field($_POST['edit_source']));
        $target = str_replace(' ', '%20', sanitize_text_field($_POST['edit_target']));

        if (!empty($source) && !empty($target)) {
            $wpdb->update($table_name, ['source_url' => $source, 'target_url' => $target], ['id' => $id]);
            $message = '<div class="updated"><p>Redirect updated successfully!</p></div>';
        }
    }

    if (isset($_POST['save_redirect'])) {
        if (check_admin_referer('add_redirect_nonce', 'msr_nonce_field')) {
            $source = str_replace(' ', '%20', sanitize_text_field($_POST['source']));
            $target = str_replace(' ', '%20', sanitize_text_field($_POST['target']));

            if (!empty($source) && !empty($target)) {
                $wpdb->insert($table_name, ['source_url' => $source, 'target_url' => $target]);
                $message = '<div class="updated"><p>Redirect added successfully!</p></div>';
            }
        }
    }

    $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
    $query = "SELECT * FROM $table_name";
    if (!empty($search)) {
        $query .= $wpdb->prepare(" WHERE source_url LIKE %s OR target_url LIKE %s", '%' . $search . '%', '%' . $search . '%');
    }
    $results = $wpdb->get_results($query . " ORDER BY id DESC");

    ?>
    <div class="wrap">
        <h1>Manage Redirects</h1>
        <?php echo $message; ?>

        <div class="postbox" style="margin-top: 20px; padding: 15px;">
            <h3>Add New Redirect</h3>
            <form method="POST">
                <?php wp_nonce_field('add_redirect_nonce', 'msr_nonce_field'); ?>
                <input type="text" name="source" placeholder="Old Path (e.g. /old url)" style="width: 25%;" required>
                <input type="text" name="target" placeholder="New URL (e.g. /new-url)" style="width: 25%;" required>
                <input type="submit" name="save_redirect" class="button button-primary" value="Add Redirect">
            </form>
        </div>

        <form method="GET" style="margin-bottom: 10px; display: flex; justify-content: flex-end;">
            <input type="hidden" name="page" value="<?php echo $page_slug; ?>">
            <input type="search" name="s" value="<?php echo esc_attr($search); ?>" placeholder="Search redirects...">
            <input type="submit" class="button" value="Search">
        </form>

        <form method="POST" id="redirect-list-form">
            <?php wp_nonce_field('bulk_redirects_nonce'); ?>
            <?php wp_nonce_field('edit_redirect_nonce'); ?>
            
            <div class="tablenav top">
                <div class="alignleft actions bulkactions">
                    <select name="bulk_action">
                        <option value="-1">Bulk Actions</option>
                        <option value="delete">Delete</option>
                    </select>
                    <input type="submit" class="button action" value="Apply">
                </div>
            </div>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <td id="cb" class="manage-column column-cb check-column"><input type="checkbox"></td>
                        <th>Source (Old)</th>
                        <th>Target (New)</th>
                        <th style="width: 150px;">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($results): foreach ($results as $row): 
                        $del_url = wp_nonce_url(admin_url('admin.php?page='.$page_slug.'&action=delete&id='.$row->id), 'delete_redirect_'.$row->id);
                    ?>
                        <tr id="row-<?php echo $row->id; ?>">
                            <th scope="row" class="check-column"><input type="checkbox" name="redirect_ids[]" value="<?php echo $row->id; ?>"></th>
                            
                            <td class="col-source"><code><a href="<?php echo esc_html($row->source_url); ?>" target="_blank"><?php echo esc_html($row->source_url); ?></a></code></td>
                            <td class="col-target"><code><a href="<?php echo esc_html($row->target_url); ?>" target="_blank"><?php echo esc_html($row->target_url); ?></code></td>
                            
                            <td class="col-edit" style="display:none;" colspan="2">
                                <input type="hidden" name="redirect_id" value="<?php echo $row->id; ?>" disabled>
                                <input type="text" name="edit_source" value="<?php echo esc_attr($row->source_url); ?>" style="width:45%" disabled>
                                <input type="text" name="edit_target" value="<?php echo esc_attr($row->target_url); ?>" style="width:45%" disabled>
                            </td>

                            <td>
                                <div class="action-buttons">
                                    <a href="#" class="edit-link" onclick="toggleEdit(<?php echo $row->id; ?>); return false;">Edit</a> | 
                                    <a href="<?php echo $del_url; ?>" style="color: #a00;" onclick="return confirm('Are you sure?')">Delete</a>
                                </div>
                                <div class="edit-buttons" style="display:none;">
                                    <input type="submit" name="update_redirect" class="button button-small button-primary" value="Save">
                                    <a href="#" class="button button-small" onclick="toggleEdit(<?php echo $row->id; ?>); return false;">Cancel</a>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="4">No redirects found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </form>
    </div>

    <script>
        document.querySelector('thead .column-cb input').onclick = function() {
            var checkboxes = document.querySelectorAll('tbody .check-column input');
            for (var checkbox of checkboxes) { checkbox.checked = this.checked; }
        }

        function toggleEdit(id) {
            const row = document.getElementById('row-' + id);
            const sourceCol = row.querySelector('.col-source');
            const targetCol = row.querySelector('.col-target');
            const editCol = row.querySelector('.col-edit');
            const actionBtns = row.querySelector('.action-buttons');
            const editBtns = row.querySelector('.edit-buttons');
            const inputs = editCol.querySelectorAll('input');

            if (editCol.style.display === 'none') {
                sourceCol.style.display = 'none';
                targetCol.style.display = 'none';
                editCol.style.display = 'table-cell';
                actionBtns.style.display = 'none';
                editBtns.style.display = 'block';
                inputs.forEach(input => input.disabled = false);
            } else {
                sourceCol.style.display = 'table-cell';
                targetCol.style.display = 'table-cell';
                editCol.style.display = 'none';
                actionBtns.style.display = 'block';
                editBtns.style.display = 'none';
                inputs.forEach(input => input.disabled = true);
            }
        }
    </script>
    <?php
}

add_action('template_redirect', 'msr_do_redirect');
function msr_do_redirect() {
    global $wpdb;
    $current_path = rtrim(rawurldecode($_SERVER['REQUEST_URI']), '/');
    if (empty($current_path)) $current_path = '/';
    $table_name = $wpdb->prefix . 'custom_redirects';

    $redirect = $wpdb->get_row($wpdb->prepare(
        "SELECT target_url FROM $table_name WHERE source_url = %s OR source_url = %s LIMIT 1",
        $current_path,
        $current_path . '/'
    ));

    if ($redirect) {
        wp_redirect($redirect->target_url, 301);
        exit;
    }
}