<?php
/**
 * Plugin Name: Visitor Data
 * Description: Tracks hits with custom date ranges.
 * Version: 1.0
 * Author: Ryan Lyons
 * Author URI: https://ryanlyons.dev/
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

register_activation_hook(__FILE__, 'sva_ultimate_db_setup');
function sva_ultimate_db_setup() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'visitor_logs';
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        time datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        ip_address varchar(45) NOT NULL,
        location varchar(150) DEFAULT 'Unknown',
        country_code varchar(10) DEFAULT '', 
        page_url text NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

add_action('wp_footer', function() {
    if (is_admin() || is_user_logged_in()) return;
    global $wpdb;
    $ip = $_SERVER['REMOTE_ADDR'];
    $geo = get_transient('sva_geo_' . $ip);
    if (false === $geo) {
        $res = wp_remote_get("http://ip-api.com/json/{$ip}?fields=status,city,country,countryCode");
        $body = !is_wp_error($res) ? json_decode(wp_remote_retrieve_body($res)) : null;
        $geo = ($body && $body->status === 'success') ? ['loc' => $body->city . ', ' . $body->country, 'code' => $body->countryCode] : ['loc' => 'Unknown', 'code' => ''];
        set_transient('sva_geo_' . $ip, $geo, DAY_IN_SECONDS);
    }
    $wpdb->insert($wpdb->prefix . 'visitor_logs', ['ip_address' => $ip, 'location' => $geo['loc'], 'country_code' => $geo['code'], 'page_url' => home_url($_SERVER['REQUEST_URI'])]);
});

add_action('admin_menu', function() {
    add_menu_page('Visitor Stats', 'Visitor Stats', 'manage_options', 'visitor-stats', 'sva_render_final_dashboard', 'dashicons-chart-line');
});

function sva_render_final_dashboard() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'visitor_logs';
    
    $view = isset($_GET['view']) ? sanitize_text_field($_GET['view']) : 'general';
    $target = isset($_GET['target']) ? sanitize_text_field($_GET['target']) : '';
    $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : date('Y-m-d', strtotime('-7 days'));
    $end_date   = isset($_GET['end_date'])   ? sanitize_text_field($_GET['end_date'])   : date('Y-m-d');
    $where_date = $wpdb->prepare("DATE(time) BETWEEN %s AND %s", $start_date, $end_date);

    echo '<div class="wrap"><h1>Visitor Analytics Dashboard</h1>';
    
    if ($view !== 'general') {
        echo '<a href="?page=visitor-stats" class="button" style="margin-bottom:20px;">&larr; Back to Overview</a>';
    }

    if ($view === 'page' && !empty($target)) {
        $hits = $wpdb->get_results($wpdb->prepare("SELECT time, location, ip_address FROM $table_name WHERE page_url = %s AND $where_date ORDER BY time DESC", $target));
        $map_stats = $wpdb->get_results($wpdb->prepare("SELECT SUBSTRING_INDEX(location, ', ', -1) as country, COUNT(*) as hits FROM $table_name WHERE page_url = %s AND $where_date GROUP BY country", $target));
        render_drilldown_layout("Details for: $target", $hits, $map_stats, 'Date/Time', 'Location & IP');
    } 
    
    elseif ($view === 'country' && !empty($target)) {
        $hits = $wpdb->get_results($wpdb->prepare("SELECT page_url as main, COUNT(*) as hits, GROUP_CONCAT(DISTINCT location SEPARATOR ', ') as sub FROM $table_name WHERE location LIKE %s AND $where_date GROUP BY page_url", '%' . $target . '%'));
        $map_stats = [(object)['country' => $target, 'hits' => 1]];
        render_drilldown_layout("Visitors from $target", $hits, $map_stats, 'Page URL', 'City Breakdown');
    } 
    
    else {
        $top_urls = $wpdb->get_col("SELECT page_url FROM $table_name WHERE $where_date GROUP BY page_url ORDER BY COUNT(*) DESC LIMIT 10");
        $dates = []; $period = new DatePeriod(new DateTime($start_date), new DateInterval('P1D'), (new DateTime($end_date))->modify('+1 day'));
        foreach ($period as $d) { $dates[] = $d->format("Y-m-d"); }

        $datasets = []; $colors = ['#2271b1', '#d63638', '#dba617', '#22a7f0', '#191e23', '#46b450', '#a36d9d', '#ff8066', '#845ec2', '#008e9b'];
        foreach ($top_urls as $i => $url) {
            $counts = [];
            foreach ($dates as $date) {
                $count = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE page_url = %s AND DATE(time) = %s", $url, $date));
                $counts[] = (int)$count;
            }
            $datasets[] = ['label' => basename($url) ?: 'Home', 'data' => $counts, 'borderColor' => $colors[$i] ?? '#ccc', 'borderWidth' => 2, 'hoverBorderWidth' => 6, 'tension' => 0.3, 'pointRadius' => 2];
        }

        $country_stats = $wpdb->get_results("SELECT SUBSTRING_INDEX(location, ', ', -1) as country, COUNT(*) as hits FROM $table_name WHERE $where_date AND location != 'Unknown' GROUP BY country");
        $map_data = [['Country', 'Hits']]; foreach ($country_stats as $s) $map_data[] = [$s->country, (int)$s->hits];
        ?>

        <div class="tablenav top" style="margin-bottom: 20px; background: #fff; padding: 10px; border: 1px solid #ccd0d4;">
            <form method="get" style="display:flex; gap:10px; align-items:center;">
                <input type="hidden" name="page" value="visitor-stats">
                <input type="date" name="start_date" value="<?php echo $start_date; ?>">
                <input type="date" name="end_date" value="<?php echo $end_date; ?>">
                <input type="submit" class="button button-primary" value="Update Range">
                <a href="?page=visitor-stats" class="button">Reset</a>
            </form>
        </div>

        <div style="background: #fff; border: 1px solid #ccd0d4; padding: 20px; margin-bottom: 20px;">
            <h3>Traffic Trends (Top 10 Pages)</h3>
            <div style="position: relative; height: 350px; width: 100%;"><canvas id="lineChart"></canvas></div>
        </div>

        <div style="display: flex; gap: 20px; flex-wrap: wrap;">
            <div style="flex: 1; min-width: 450px; background: #fff; border: 1px solid #ccd0d4; padding: 20px;">
                <h3>Global Heat Map <small>(Click country to filter)</small></h3>
                <div id="regions_div" style="width: 100%; height: 350px;"></div>
            </div>
            
            <div style="flex: 1; min-width: 450px; background: #fff; border: 1px solid #ccd0d4; padding: 20px;">
                <h3>Top Pages <small>(Click row for history)</small></h3>
                <table class="wp-list-table widefat fixed striped" style="border:none;">
                    <thead><tr><th>Page URL</th><th>Hits</th></tr></thead>
                    <tbody>
                        <?php if($top_urls): foreach ($top_urls as $url): 
                            $total = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE page_url = %s AND $where_date", $url));
                        ?>
                        <tr style="cursor:pointer;" onclick="window.location='?page=visitor-stats&view=page&target=<?php echo urlencode($url); ?>'">
                            <td><code style="font-size:11px;"><?php echo esc_html($url); ?></code></td>
                            <td><strong><?php echo $total; ?></strong></td>
                        </tr>
                        <?php endforeach; else: echo '<tr><td colspan="2">No data found.</td></tr>'; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <script type="text/javascript" src="https://www.gstatic.com/charts/loader.js"></script>
        <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
        <script>
            
            new Chart(document.getElementById('lineChart'), {
                type: 'line',
                data: { labels: <?php echo json_encode($dates); ?>, datasets: <?php echo json_encode($datasets); ?> },
                options: { 
                    maintainAspectRatio: false, 
                    interaction: { mode: 'dataset', intersect: false },
                    plugins: { legend: { position: 'bottom', labels: { usePointStyle: true, boxWidth: 6 } } } 
                }
            });

            google.charts.load('current', {'packages':['geochart']});
            google.charts.setOnLoadCallback(() => {
                var data = google.visualization.arrayToDataTable(<?php echo json_encode($map_data); ?>);
                var chart = new google.visualization.GeoChart(document.getElementById('regions_div'));
                google.visualization.events.addListener(chart, 'select', () => {
                    var s = chart.getSelection();
                    if (s.length > 0) window.location = '?page=visitor-stats&view=country&target=' + encodeURIComponent(data.getValue(s[0].row, 0));
                });
                chart.draw(data, {colorAxis: {colors: ['#f0f8ff', '#2271b1']}, backgroundColor: 'transparent', datalessRegionColor: '#f0f0f0'});
            });
        </script>
        <?php
    }
    echo '</div>';
}

function render_drilldown_layout($title, $rows, $map_db, $c1, $c2) {
    $map_data = [['Country', 'Hits']]; foreach ($map_db as $s) $map_data[] = [$s->country, (int)$s->hits];
    ?>
    <div style="background: #fff; border: 1px solid #ccd0d4; padding: 20px;">
        <h2><?php echo esc_html($title); ?></h2>
        <div id="drill_map" style="width: 100%; height: 350px; margin-bottom: 20px; background:#f9f9f9;"></div>
        <table class="wp-list-table widefat fixed striped">
            <thead><tr><th><?php echo $c1; ?></th><th><?php echo $c2; ?></th></tr></thead>
            <tbody>
                <?php foreach ($rows as $r): ?>
                <tr>
                    <td><?php echo isset($r->time) ? $r->time : esc_html($r->main); ?></td>
                    <td><?php echo isset($r->location) ? esc_html($r->location . " (" . $r->ip_address . ")") : "<strong>{$r->hits} hits</strong> — " . esc_html($r->sub); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <script type="text/javascript" src="https://www.gstatic.com/charts/loader.js"></script>
    <script>
        google.charts.load('current', {'packages':['geochart']});
        google.charts.setOnLoadCallback(() => {
            var data = google.visualization.arrayToDataTable(<?php echo json_encode($map_data); ?>);
            new google.visualization.GeoChart(document.getElementById('drill_map')).draw(data, {colorAxis: {colors: ['#f0f8ff', '#2271b1']}});
        });
    </script>
    <?php
}